//----------------------------------------------------------------------------
//
// Copyright (C) Sartorius Stedim Data Analytics AB 2017 -
//
// Use, modification and distribution are subject to the Boost Software
// License, Version 1.0. (See http://www.boost.org/LICENSE_1_0.txt)
//
//----------------------------------------------------------------------------

#include "stdafx.h"
#ifdef _WIN32
#include <windows.h>
#endif

#include "EzQ.h"
#include "utf8util.h"
#include <memory>
#include <functional>

void CreateDummyPredictionSet(SQ_Project hProject, int iModelIndex, float*& fQuantMatrix, int& nQuantRows, int& nQuantCols, char**& szQualMatrix, int& nQualRows, int& nQualCols, char**& szNamesForPredict);
void PrintMatrix(const CEzFloatMatrix& fMatrix);

template <typename T>
void MyDelete(T* f) {
   free(f);
}

//

template<typename T>
using deleted_unique_ptr = std::unique_ptr<T, std::function<void(T*)>>;

int _tmain(int argc, TCHAR* argv[], TCHAR* /*envp[]*/)
{
   deleted_unique_ptr<char> project = nullptr;

#ifdef _WIN32
   if (argc < 2)
   {
      // open a file name
      OPENFILENAME ofn;
      char szFile[260];       // buffer for file name
      ZeroMemory(&ofn, sizeof(ofn));
      ofn.lStructSize = sizeof(ofn);
      ofn.hwndOwner = nullptr;
      ofn.lpstrFile = szFile;
      ofn.lpstrFile[0] = '\0';
      ofn.nMaxFile = sizeof(szFile);
      ofn.lpstrFilter = _T("usp\0*.usp");
      ofn.nFilterIndex = 0;
      ofn.lpstrFileTitle = nullptr;
      ofn.nMaxFileTitle = 0;
      ofn.lpstrInitialDir = nullptr;
      ofn.Flags = OFN_PATHMUSTEXIST | OFN_FILEMUSTEXIST;

      if (GetOpenFileName(&ofn) == 0)
         return 0;

      project = deleted_unique_ptr<char>(TCHARtoUTF8(ofn.lpstrFile), MyDelete<char>);
   }
#endif // _WIN32

   if (argc != 2 && project == nullptr)
   {
      _ftprintf(stderr, _T("\n%s Usage:\n\n"
         ".\\SQPC++Sample.exe <USP file>\n\n"
         "    USP file            ---  is a SIMCA project file\n\n"),
         argv[0]);
      return 1;
   }

   // The Q object
   CEzQ oEzQ;

   // Result matrices
   CEzVectorData oTPS;
   CEzVectorData oDModXPS;
   CEzFloatMatrix fmatData;

   // Dummy prediction set variables
   float* fmatQuantitativeData = nullptr;
   int nQuantRows = 0, nQuantCols = 0;
   char** szMatQualitativeData = nullptr;
   char** szNamesForPredict = nullptr;

   int nQualRows = 0, nQualCols = 0;
   SQ_ErrorCode eError;

   // Open the project
   if (!project)
   {
      project = deleted_unique_ptr<char>(TCHARtoUTF8(argv[1]), MyDelete<char>);
   }
   eError = oEzQ.OpenProject(project.get(), nullptr);

   if (eError != SQ_E_OK)
   {
      deleted_unique_ptr<TCHAR> errorstr(UTF8toTCHAR(oEzQ.GetErrorDescription(eError).c_str()), MyDelete<TCHAR>);
      _tprintf(_T("Could not open project. Error %s"), errorstr.get());
      return 0;
   }

   // Select which model to use, this takes the second one
   int iNumModels;
   SQ_GetNumberOfModels(oEzQ.GetProjectHandle(), &iNumModels);
   int iTestModel = iNumModels > 1 ? 2 : 1;
   eError = oEzQ.SetModel(iTestModel);
   if (eError != SQ_E_OK)
   {
      deleted_unique_ptr<TCHAR> errorstr(UTF8toTCHAR(oEzQ.GetErrorDescription(eError).c_str()), MyDelete<TCHAR>);
      _tprintf(_T("Could not open model. Error %s"), errorstr.get());
      return 0;
   }

   // Create dummy prediction set
   CreateDummyPredictionSet(oEzQ.GetProjectHandle(), iTestModel, fmatQuantitativeData, nQuantRows, nQuantCols, szMatQualitativeData, nQualRows, nQualCols, szNamesForPredict);

   // Predict with dummy prediction set.
   if (true)
   {
      // "Predict" checks variable names so that we know that the correct prediction are made if we cannot guarantee that the names in the model are always the same.
      eError = oEzQ.Predict(fmatQuantitativeData, nQuantRows, nQuantCols, szNamesForPredict);
   }
   else
   {
      // "Predict2" handles also qualitative variables, but assumes all data is in the correct order already, no name checking.
      eError = oEzQ.Predict2(fmatQuantitativeData, nQuantRows, nQuantCols, szMatQualitativeData, nQualRows, nQualCols);
   }



   if (eError != SQ_E_OK)
   {
      deleted_unique_ptr<TCHAR> errorstr(UTF8toTCHAR(oEzQ.GetErrorDescription(eError).c_str()), MyDelete<TCHAR>);
      _tprintf(_T("Could not predict. Error %s"), errorstr.get());
      return 0;
   }

   // Delete dummy prediction set
   if (fmatQuantitativeData)
      delete[] fmatQuantitativeData;
   if (szMatQualitativeData)
   {
      for (int i = 0; i < nQualCols * nQualRows; ++i)
         delete[] szMatQualitativeData[i];
      delete[] szMatQualitativeData;
   }

   // Get some results
   oEzQ.GetTPS(oTPS);
   oEzQ.GetDModXPS(oDModXPS);


   // Print results
   oTPS.GetMatrix(fmatData);
   _tprintf(_T("Score:\n"));
   PrintMatrix(fmatData);
   oDModXPS.GetMatrix(fmatData);
   _tprintf(_T("DModX:\n"));
   PrintMatrix(fmatData);

   CEzVectorData oDModYPS;
   oEzQ.GetDModYPS(oDModYPS);
   oDModYPS.GetMatrix(fmatData);
   _tprintf(_T("DModY:\n"));
   PrintMatrix(fmatData);

   return 0;
}

/**
 * Function to print all values in a matrix
 */
void PrintMatrix(const CEzFloatMatrix& fMatrix)
{
   int nRows, nCols;
   float fVal;

   SQ_GetNumRowsInFloatMatrix(fMatrix, &nRows);
   SQ_GetNumColumnsInFloatMatrix(fMatrix, &nCols);

   for (int iRow = 1; iRow <= nRows; ++iRow)
   {
      for (int iCol = 1; iCol <= nCols; ++iCol)
      {
         SQ_GetDataFromFloatMatrix(fMatrix, iRow, iCol, &fVal);
         _tprintf(_T("%f\t"), fVal);
      }
      _tprintf(_T("\n"));
   }
}

/**
* Function to create a dummy prediction set. In a "real" world example it is read from a file or somewhere else.
*/
void CreateDummyPredictionSet(SQ_Project hProject, int iModelIndex, float*& fQuantMatrix, int& nQuantRows, int& nQuantCols, char**& szQualMatrix, int& nQualRows, int& nQualCols, char**& szNamesForPredict)
{
   int iNumVars;
   int iModelNumber;
   SQ_Model hModel = nullptr;
   SQ_PreparePrediction hPrepPred = nullptr;
   SQ_VariableVector hVarVec = nullptr;
   SQ_StringVector oVarNames = nullptr;
   SQ_Variable hVariable = nullptr;
   char szBuffer[256];
   int iNumObs = 2; // Create a prediction set with only one row

   fQuantMatrix = nullptr;
   szQualMatrix = nullptr;

   SQ_GetModelNumberFromIndex(hProject, iModelIndex, &iModelNumber);
   SQ_GetModel(hProject, iModelNumber, &hModel);
   SQ_GetPreparePrediction(hModel, &hPrepPred);
   SQ_GetVariablesForPrediction(hPrepPred, &hVarVec);
   SQ_GetNumVariablesInVector(hVarVec, &iNumVars);

   // Count number of qualitative and quantitative variables
   int nQualitative = 0;
   for (int iVar = 1; iVar <= iNumVars; ++iVar)
   {
      SQ_Bool bIsQual;
      SQ_GetVariableFromVector(hVarVec, iVar, &hVariable);
      SQ_IsQualitative(hVariable, &bIsQual);
      if (bIsQual == SQ_True)
         ++nQualitative;
   }

   // Create prediction data
   nQualCols = nQualitative;
   nQualRows = iNumObs;
   if (nQualitative > 0)
      szQualMatrix = new char* [nQualCols * nQualRows];

   nQuantCols = iNumVars - nQualitative;
   nQuantRows = iNumObs;
   fQuantMatrix = new float[nQuantCols * nQuantRows];
   szNamesForPredict = new char* [iNumVars];

   int iQualPos = 0;
   int iQuantPos = 0;
   SQ_Bool bIsQual;

   for (int iVar = 1; iVar <= iNumVars; ++iVar)
   {
      SQ_GetVariableFromVector(hVarVec, iVar, &hVariable);
      SQ_IsQualitative(hVariable, &bIsQual);

      char szName[100];
      SQ_GetVariableName(hVariable, 1, szName, 100);
      szNamesForPredict[iVar - 1] = _strdup(szName);

      for (int iRow = 0; iRow < iNumObs; ++iRow)
      {
         if (bIsQual == SQ_True)
         {
            SQ_GetQualitativeSettings(hVariable, &oVarNames);
            SQ_GetStringFromVector(oVarNames, 1, szBuffer, sizeof(szBuffer)); // Just use the first qualitative setting
            szQualMatrix[iRow * nQualCols + iQualPos] = _strdup(szBuffer);
         }
         else
         {
            fQuantMatrix[iRow * nQuantCols + iQuantPos] = float(iRow + iVar); // Fill with some data
         }
      }

      if (bIsQual == SQ_True)
         iQualPos++;
      else
         iQuantPos++;
   }
}
